<?php
/**
 * Plugin Name: cmercury Plugin
 */

if (!defined('ABSPATH')) exit;

define('CMERCURY_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('CMERCURY_PLUGIN_URL', plugin_dir_url(__FILE__));
define('CMERCURY_API_BASE', 'https://api.cmercury.com/api');

// Load class files
require_once CMERCURY_PLUGIN_DIR . 'includes/class-cmercury-api.php';
require_once CMERCURY_PLUGIN_DIR . 'includes/class-cmercury-hooks.php';
require_once CMERCURY_PLUGIN_DIR . 'includes/class-cmercury-mailer.php';
require_once CMERCURY_PLUGIN_DIR . 'admin/settings-page.php';
require_once CMERCURY_PLUGIN_DIR . 'includes/class-cmercury-woocommerce.php';

// initialize *once*
add_action('plugins_loaded', function() {
    Cmercury_Hooks::init();
    Cmercury_Mailer::init();
    Cmercury_WooCommerce::init();
});


    // admin menu
    add_action('admin_menu', function() {
        add_menu_page(
            'cmercury Settings',
            'cmercury',
            'manage_options',
            'cmercury-settings',
            'cmercury_render_settings_page',
            plugins_url('assets/cmercury-menu-logo.png', __FILE__),
            56
        );
    });

    // cron schedules
    add_filter('cron_schedules', function($schedules) {
        $schedules['cmercury_every_5_minutes'] = [
            'interval' => 300,
            'display'  => __('Every 5 minutes')
        ];
        $schedules['cmercury_every_10_minutes'] = [
            'interval' => 600,
            'display'  => __('Every 10 minutes')
        ];
        $schedules['cmercury_every_15_minutes'] = [
            'interval' => 900,
            'display'  => __('Every 15 minutes')
        ];
        $schedules['cmercury_every_30_minutes'] = [
            'interval' => 1800,
            'display'  => __('Every 30 minutes')
        ];
        return $schedules;
    });


    //log file configuration
    if (!function_exists('cmercury_log')) {
        function cmercury_log($message) {
            // Check if logging enabled
            if (!get_option('cmercury_logging_enabled')) {
                return;
            }
    
            // Build log file path
            $upload_dir = wp_upload_dir();
    		$date_suffix = date('Ymd'); // Generates date in 20251125 format
    		$log_file = trailingslashit($upload_dir['basedir']) . "cmercury-wordpress-plugin-{$date_suffix}.log";        
            
            // Convert array/object to string
            if (is_array($message) || is_object($message)) {
                $message = print_r($message, true);
            }
    
            // Construct log entry
            $line = '[' . date('Y-m-d H:i:s') . '] ' . $message . PHP_EOL;
    
            // Write
            file_put_contents($log_file, $line, FILE_APPEND);
        }
    }

    register_activation_hook( __FILE__, function() {Cmercury_WooCommerce::maybe_schedule_cron();});

    add_action('update_option_cmercury_wc_sync_freq', function( $old, $new ) {
        if ( $old !== $new ) {
            Cmercury_WooCommerce::reschedule_cron( $new );	}
        }, 
        10, 2 );

    //export customers as csv
    add_action('admin_post_cmercury_wc_export_csv', ['Cmercury_WooCommerce', 'handle_csv_export']);

    function cmercury_show_contextual_saved_notice() {
        // 1. Check permissions
        if (!current_user_can('manage_options') ) return;
    
        // 2. Check if we are on the correct settings page and a save has occurred
        if ( isset($_GET['page']) && $_GET['page'] === 'cmercury-settings' && isset($_GET['settings-updated']) ) {
    
            // Determine the current tab (defaults to 'authentication' if no tab is specified)
            $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'authentication';
    
            // Map tab slugs to the custom, user-friendly message
            $tab_messages = [
                'authentication' => 'API Authentication settings saved successfully.',
                'contacts'          => 'Forms Integration settings saved successfully.',
                'trigger'  => 'Email Delivery settings saved successfully.',
                'woocommerce'    => 'WooCommerce Sync settings saved successfully.',
            ];
    
            // Get the appropriate message, defaulting to a generic one if the tab is unrecognized
            $message = $tab_messages[$current_tab] ?? 'Settings saved successfully.';
    
            // Output the notice
            echo '<div class="notice notice-success is-dismissible"><p><strong>' . esc_html($message) . '</strong></p></div>';
        }
    }

    add_action('admin_notices', 'cmercury_show_contextual_saved_notice');

    //Update notification
    
    // 1) Define plugin version constant (if not already present)
    if (! defined('CMERCURY_PLUGIN_VERSION')) {
        // set this manually to match plugin header version with every update
        define('CMERCURY_PLUGIN_VERSION', '1.1.0');
    }
    
    // 2) Config: remote JSON URL and transient key
    if (! defined('CMERCURY_UPDATE_JSON_URL')) {
        define('CMERCURY_UPDATE_JSON_URL', 'https://cmercury.com/wp-content/uploads/cmercury-wp-plugin-update.json');
    }
    if (! defined('CMERCURY_UPDATE_TRANSIENT')) {
        define('CMERCURY_UPDATE_TRANSIENT', 'cmercury_remote_update_info');
    }
    if (! defined('CMERCURY_UPDATE_TRANSIENT_TTL')) {
        // cache remote JSON for 12 hours (adjust if you want more/less)
        define('CMERCURY_UPDATE_TRANSIENT_TTL', 360 * MINUTE_IN_SECONDS);
    }

    // 3) Hook up admin init hooks
    add_action('admin_init', 'cmercury_maybe_check_remote_update');
    add_action('admin_notices', 'cmercury_show_update_notice');
    add_action('wp_ajax_cmercury_dismiss_update_notice', 'cmercury_ajax_dismiss_update_notice');

    /**
     * Fetch remote JSON and cache it in a transient.
     * Returns decoded associative array or null on failure.
     */
    function cmercury_get_remote_update_info() {
    $cached = get_transient(CMERCURY_UPDATE_TRANSIENT);
    if ($cached !== false) {
        return $cached;
    }

    // safe default
    $result = null;
    
    //pass the current version
    $url = add_query_arg(
        [ 'installedversion' => CMERCURY_PLUGIN_VERSION ],
        CMERCURY_UPDATE_JSON_URL
    );

    $resp = wp_remote_get($url, [
        'timeout' => 8,
        'headers' => [
            'Accept' => 'application/json'
        ]
    ]);

    if (is_wp_error($resp)) {
        // cache empty negative result for short time to avoid hammering
        set_transient(CMERCURY_UPDATE_TRANSIENT, null, 360 * MINUTE_IN_SECONDS);
        return null;
    }

    $code = wp_remote_retrieve_response_code($resp);
    $body = wp_remote_retrieve_body($resp);

    if ($code === 200 && $body) {
        $decoded = json_decode($body, true);
        if (is_array($decoded) && ! empty($decoded['version'])) {
            // normalize data keys we will later use
            $result = [
                'version'     => (string) $decoded['version'],
                'title'       => isset($decoded['title']) ? sanitize_text_field($decoded['title']) : 'New plugin update available',
                'message'     => isset($decoded['message']) ? wp_kses_post($decoded['message']) : '',
                'download_url'=> isset($decoded['download_url']) ? esc_url_raw($decoded['download_url']) : '',
                'important'   => ! empty($decoded['important'])
            ];

            // store for TTL
            set_transient(CMERCURY_UPDATE_TRANSIENT, $result, CMERCURY_UPDATE_TRANSIENT_TTL);
            return $result;
        }
    }

    // fallback: cache negative for short time
    set_transient(CMERCURY_UPDATE_TRANSIENT, null, 15 * MINUTE_IN_SECONDS);
    return null;
}
    
    /**
     * Called on admin_init — small helper if you want to proactively trigger fetch.
     * We keep it cheap; it simply warms the transient for the current admin request.
     */
    function cmercury_maybe_check_remote_update() {
        // Only for users who can manage options (admins)
        if (! current_user_can('manage_options')) return;
        // Kick off fetch in a non-blocking way? We'll just warm the transient (fast).
        cmercury_get_remote_update_info();
    }

    /**
     * Render update admin notice (only when remote version > installed and not dismissed).
     */
    function cmercury_show_update_notice() {
    if (! current_user_can('manage_options')) return;

    $remote = cmercury_get_remote_update_info();
    if (empty($remote) || empty($remote['version'])) return;

    // get installed version — prefer constant, else fallback to plugin header if you want
    $installed_version = defined('CMERCURY_PLUGIN_VERSION') ? CMERCURY_PLUGIN_VERSION : '0.0.0';
    $remote_version = $remote['version'];

    // Only show if remote version strictly greater than installed
    if (! version_compare($remote_version, $installed_version, '>')) {
        return;
    }

    // Has the current user dismissed this remote version already?
    $user_id = get_current_user_id();
    $dismissed_for = get_user_meta($user_id, 'cmercury_update_dismissed_for', true);
    if ($dismissed_for && version_compare($dismissed_for, $remote_version, '>=')) {
        // already dismissed this version or a later one
        return;
    }

    // Build HTML for notice
    $title = esc_html($remote['title']);
    $message = wp_kses_post($remote['message']);
    $download = ! empty($remote['download_url']) ? esc_url($remote['download_url']) : '';

    // Nonce for AJAX dismiss
    $nonce = wp_create_nonce('cmercury_dismiss_update');

    // choose notice class (important => error, else updated)
    $notice_class = ! empty($remote['important']) ? 'notice notice-warning is-dismissible' : 'notice notice-info is-dismissible';

    ?>
    <div id="cmercury-update-notice" class="<?php echo esc_attr($notice_class); ?>">
        <p style="font-weight:600;margin:0;"><?php echo $title; ?></p>
        <?php if ($message): ?>
            <p style="margin:.3em 0;"><?php echo $message; ?></p>
        <?php endif; ?>
        <?php if ($download): ?>
            <p style="margin:.3em 0;">
                <a class="button button-primary" href="<?php echo $download; ?>" target="_blank">Download update</a>
            </p>
        <?php endif; ?>
        <p style="margin-top:.5em;">
            <a href="#" class="cmercury-dismiss-update" data-nonce="<?php echo esc_attr($nonce); ?>" data-version="<?php echo esc_attr($remote_version); ?>">Dismiss</a>
        </p>
    </div>

    <script>
    (function(){
        // attach click to dismiss link (AJAX)
        var links = document.querySelectorAll('.cmercury-dismiss-update');
        Array.prototype.forEach.call(links, function(el){
            el.addEventListener('click', function(e){
                e.preventDefault();
                var nonce = el.getAttribute('data-nonce');
                var version = el.getAttribute('data-version');

                var data = new URLSearchParams();
                data.append('action', 'cmercury_dismiss_update_notice');
                data.append('nonce', nonce);
                data.append('version', version);

                fetch(ajaxurl, {
                    method: 'POST',
                    credentials: 'same-origin',
                    headers: {'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8'},
                    body: data.toString()
                }).then(function(r){ return r.json(); })
                .then(function(j){
                    if (j && j.success) {
                        var notice = document.getElementById('cmercury-update-notice');
                        if (notice) notice.style.display = 'none';
                    } else {
                        alert(j.data && j.data.message ? j.data.message : 'Could not dismiss');
                    }
                }).catch(function(){
                    alert('Could not dismiss (network error)');
                });

            }, false);
        });
    })();
    </script>
    <?php
}
    
    /**
     * AJAX: dismiss update notice for the current user and version.
     * Stores user meta 'cmercury_update_dismissed_for' => version string
     */
    function cmercury_ajax_dismiss_update_notice() {
        // capability check
        if (! current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized'], 403);
        }
    
        $nonce = isset($_POST['nonce']) ? sanitize_text_field($_POST['nonce']) : '';
        $version = isset($_POST['version']) ? sanitize_text_field($_POST['version']) : '';
    
        if (! wp_verify_nonce($nonce, 'cmercury_dismiss_update')) {
            wp_send_json_error(['message' => 'Invalid nonce'], 400);
        }
    
        if (! $version) {
            wp_send_json_error(['message' => 'Missing version'], 400);
        }
    
        $user_id = get_current_user_id();
        update_user_meta($user_id, 'cmercury_update_dismissed_for', $version);
    
        wp_send_json_success(['message' => 'Dismissed']);
    }
    // end of update notification.

    // uninstall cleanup
    function cmercury_plugin_uninstall() {
        
    	$options = [
            'cmercury_api_token'
            ,'cmercury_cf7_email_field'
            ,'cmercury_default_subject'
            ,'cmercury_enable_mailer'
            ,'cmercury_forms_autointegrate'
            ,'cmercury_from_email'
            ,'cmercury_gravity_email_field'
            ,'cmercury_list_id'
            ,'cmercury_logging_enabled'
            ,'cmercury_mailer_fallback'
            ,'cmercury_reply_to'
            ,'cmercury_sender_name'
            ,'cmercury_trigger_id'
            ,'cmercury_wc_attribute_map'
            ,'cmercury_wc_auto_create_list'
            ,'cmercury_wc_auto_sync'
            ,'cmercury_wc_from_email'
            ,'cmercury_wc_from_name'
            ,'cmercury_wc_last_order_id'
            ,'cmercury_wc_last_sync'
            ,'cmercury_wc_last_user_id'
            ,'cmercury_wc_list_id'
            ,'cmercury_wc_sync_done'
            ,'cmercury_wc_sync_running'
            ,'cmercury_wc_sync_total'
            ,'cmercury_wpforms_email_field'
			,'cmercury_api_token_status'
        ];
    
        foreach ($options as $option) { 
            delete_option($option);
        }
    }

//Uninstall
register_uninstall_hook(__FILE__, 'cmercury_plugin_uninstall');