<?php
if ( ! defined('ABSPATH') ) exit;

class Cmercury_Api {
    private $token;
    private $base;

    public function __construct() {
        $this->token = trim( get_option('cmercury_api_token', '') );
        $this->base = rtrim( CMERCURY_API_BASE, '/' ) . '/';
    }

 //add or update contact 
    public function add_contact( $listId, $email, $meta = [] ) {

    cmercury_log("Calling Add Contact API");


    if ( empty( $this->token ) ) {
         cmercury_log("API Key not set");
        return new WP_Error('no_token', 'API Key not set');
    }

    $payload = array_merge([
        'ListId'        => intval($listId),
        'Email'         => $email,
        'ContactStatus' => 'Active'
    ], $meta);

    $args = [
        'headers' => [
            'Content-Type' => 'application/json',
            'X-Token'      => $this->token,
            'Accept'       => 'application/json'
        ],
        'body'    => wp_json_encode($payload),
        'timeout' => 15
    ];

    $endpoint = $this->base . 'Contact/AddContact';

    // ---- Call AddContact ----
   cmercury_log("Add Contact : " .$email);

    $resp = wp_remote_post( $endpoint, $args );

    if ( is_wp_error( $resp ) ) {
        cmercury_log("Error Add Contact Remote Post: ".$resp->get_error_message());
        return $resp;
    }

    $code   = wp_remote_retrieve_response_code( $resp );
    $body   = wp_remote_retrieve_body( $resp );
    $result = json_decode($body, true);

    cmercury_log("AddContact response HTTP={$code}");
    cmercury_log($result);

    // --- If success & NOT "Contact already Exists", return normally ---
    if ( $code >= 200 && $code < 300 ) {

        // Check for "already exists"
        if (
            is_array($result) &&
            !empty($result['Value']) &&
            (
                stripos($result['Value'], 'already') !== false ||
                stripos($result['Value'], 'exist') !== false
                )
            )
    {
            cmercury_log("Contact exists → Attempting UpdateContact");

            // Call update with same payload
            return $this->update_contact($listId, $email, $meta);
        }

        return $result ?: ['raw' => $body];
    }
    elseif ( $code > 300) {
        cmercury_log("Error HTTP Add Contact >300: $code: $body");
        
    }
    
    return new WP_Error('api_error', "HTTP $code: $body");
}
   
    
    //update contact api
    public function update_contact( $listId, $email, $meta = [] ) {
         cmercury_log("Preparing Update Contact API");
        if ( empty( $this->token ) ) return new WP_Error('no_token','API token not set');
        $endpoint = $this->base . 'Contact/UpdateContact';
        $payload = array_merge( [
            'ListId' => intval( $listId ),
            'Email' => $email,
            'ContactStatus' => 'Active'
        ], $meta );
        $args = [
            'headers' => [
                'Content-Type' => 'application/json',
                'X-Token' => $this->token,
                'Accept' => 'application/json'
            ],
            'body' => wp_json_encode($payload),
            'timeout' => 15
        ];
       
        // cmercury_log(wp_json_encode($payload));
        $resp = wp_remote_post( $endpoint, $args );
        if ( is_wp_error( $resp ) ) {
            cmercury_log("Error HTTP Update Contact Remote Post: ". $resp->get_error_message());
            return $resp;}
        
        $code = wp_remote_retrieve_response_code( $resp );
        cmercury_log($code);
        $body = wp_remote_retrieve_body( $resp );
       
        //cmercury_log(json_decode($body, true) );
        if ( $code >= 200 && $code < 300 ) {
             
            
            return json_decode($body, true) ?: array('raw'=>$body);
            
        }
        elseif ($code > 300) {
            cmercury_log("Error HTTP Update Contact: $code: $body");
        }
        
       
        return new WP_Error('api_error', "HTTP $code: $body");
    }

    public function fire_trigger( $to, $triggerId, $subject, $html = '', $text = '', $headers = array() ) {
         cmercury_log("Preparing Transactional API call");
        if ( empty( $this->token ) ) return new WP_Error('no_token','API token not set');
        if ( empty( $triggerId ) ) return new WP_Error('no_trigger','Trigger ID not set');
        $endpoint = $this->base . 'Trigger/FireTrigger/v2';
        $recipients = is_array($to) ? $to : array_map('trim', explode(',', $to));
        $results = array();
        foreach ( $recipients as $recipient ) {
            if ( ! is_email( $recipient ) ) { $results[] = new WP_Error('invalid_email','Invalid: '.$recipient); continue; }
            $payload = array(
                'Email' => $recipient,
                'TriggerId' => intval( $triggerId ),
                'SenderName' => isset($headers['SenderName']) ? $headers['SenderName'] : get_option('cmercury_sender_name',''),
                'FromEmailAddress' => isset($headers['FromEmailAddress']) ? $headers['FromEmailAddress'] : get_option('cmercury_from_email',''),
                'subject' => $subject,
                'HTML' => $html,
                'Text_Message' => $text,
                'ReplyTo' => isset($headers['ReplyTo']) ? $headers['ReplyTo'] : get_option('cmercury_reply_to','')
            );
            $args = array(
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'X-Token' => $this->token,
                    'Accept' => 'application/json'
                ),
                'body' => wp_json_encode($payload),
                'timeout' => 20
            );
            
           // cmercury_log(wp_json_encode($payload));
            $resp = wp_remote_post( $endpoint, $args );
            if ( is_wp_error( $resp ) ) { $results[] = $resp; continue; }
            $code = wp_remote_retrieve_response_code( $resp );
            //cmercury_log($code);
            $body = wp_remote_retrieve_body( $resp );
           // cmercury_log(json_decode($body, true) );
            if ( $code >= 200 && $code < 300 ) $results[] = json_decode($body,true) ?: array('raw'=>$body);
            else $results[] = new WP_Error('api_error', "HTTP $code: $body");
        }
        return $results;
    }
	
public function create_mailing_list($from, $listName, $senderName) {
    cmercury_log("Preparing create List API call");

    $endpoint = $this->base . 'list/CreateMailingList';
    $payload  = [
        'FromEmailAddress' => $from,
        'ListName'         => $listName,
        'SenderName'       => $senderName
    ];

    $resp = wp_remote_post($endpoint, [
        'headers' => $this->headers(),
        'body'    => wp_json_encode($payload)
    ]);

    if (is_wp_error($resp)) {
        cmercury_log("Error Add List: ".$resp->get_error_message());
        return $resp;
    }

    $code = wp_remote_retrieve_response_code($resp);
    $body = wp_remote_retrieve_body($resp);
    $decoded = json_decode($body, true);

    //cmercury_log($code);
   // cmercury_log($decoded);

    if ($code !== 200 || !is_array($decoded)) {
        return new WP_Error('create_list_failed', 'Unexpected response while creating list');
    }

    // server response:
    // {
    //   "Value": "Mailing List Successfully Created. List ID: 11111",
    //   "HasWarning": false,
    //   "Success": true,
    //   "Status": 200
    // }

    $listId = null;

    if (!empty($decoded['Value'])) {
        // Extract List ID using regex
        if (preg_match('/List ID:\s*(\d+)/i', $decoded['Value'], $m)) {
            $listId = $m[1];
        }
    }

    if (!$listId) {
        return new WP_Error(
            'missing_listid',
            'List created but List ID not returned in expected format'
        );
    }

    return [
        'ListId'   => $listId,
        'Raw'      => $decoded
    ];
}

	
public function query_lists() {
    cmercury_log("Preparing Query List API call");

    $endpoint = $this->base . 'list/QueryList';

    $resp = wp_remote_post($endpoint, [
        'headers' => $this->headers(),
        'body'    => wp_json_encode(new stdClass())
    ]);

    if (is_wp_error($resp)) {
        return $resp;
    }

    $code = wp_remote_retrieve_response_code($resp);
   // cmercury_log($code);

    $body    = wp_remote_retrieve_body($resp);
    $decoded = json_decode($body, true);

  //  cmercury_log($decoded);

    if ($code !== 200 || !is_array($decoded)) {
        return new WP_Error('query_lists_failed', 'Invalid response from query_lists');
    }

    /**
     * Expected Response:
     * {
     *   "Value": [ {...}, {...} ],
     *   "HasWarning": false,
     *   "Success": true,
     *   "Status": 200
     * }
     */

    if (isset($decoded['Value']) && is_array($decoded['Value'])) {
        return $decoded['Value'];    // return the list array
    }

    return new WP_Error(
        'missing_list_value',
        'Unexpected format: List data missing'
    );
}


	
public function query_attributes($listId) {
    cmercury_log("Preparing Query all attributes API call");

    $endpoint = $this->base . 'Attribute/QueryAll';
    $payload  = ['ListId' => intval($listId)];

  //  cmercury_log(wp_json_encode($payload));

    $resp = wp_remote_post($endpoint, [
        'headers' => $this->headers(),
        'body'    => wp_json_encode($payload)
    ]);

    if (is_wp_error($resp)) {
        return $resp;
    }

    $code = wp_remote_retrieve_response_code($resp);
   // cmercury_log($code);

    $body = wp_remote_retrieve_body($resp);
    $decoded = json_decode($body, true);

  //  cmercury_log($decoded);

    if ($code !== 200 || !is_array($decoded)) {
        return new WP_Error('query_attributes_failed', 'Invalid response from query_attributes');
    }

    // if wrapped: { Attributes: [...] }
    if (isset($decoded['Attributes']) && is_array($decoded['Attributes'])) {
        return $decoded['Attributes'];
    }

    // if raw array
    return $decoded;
}

	
	
public function add_attribute($name, $typeId = 1) {
    cmercury_log("Preparing add attribute API call");

    $endpoint = $this->base . 'Attribute/Add';
    $payload  = [
        'Name'   => $name,
        'TypeId' => intval($typeId)
    ];

    $resp = wp_remote_post($endpoint, [
        'headers' => $this->headers(),
        'body'    => wp_json_encode($payload)
    ]);

    if (is_wp_error($resp)) {
        return $resp;
    }

    $code   = wp_remote_retrieve_response_code($resp);
    $body   = wp_remote_retrieve_body($resp);
    $decoded = json_decode($body, true);


    if ($code !== 200 || !is_array($decoded)) {
        return new WP_Error('add_attribute_failed', 'Unexpected response while adding attribute');
    }

    /**
     * Expected response:
     * {
     *   "Value": "Attribute added Successfully",
     *   "HasWarning": false,
     *   "Success": true,
     *   "Status": 200
     * }
     */

    if (!isset($decoded['Success']) || !$decoded['Success']) {
        return new WP_Error('add_attribute_failed', $decoded['Value'] ?? 'Unknown error');
    }

    // API does NOT return Id
    return [
        'success' => true,
        'message' => $decoded['Value'] ?? '',
        'raw'     => $decoded
    ];
}


	
public function enable_attribute_for_list($listId, $attributeId) {
    cmercury_log("Preparing enable attribute API call");

    $endpoint = $this->base . 'Attribute/Enable';
    $payload  = [
        'ListId' => intval($listId),
        'Id'     => intval($attributeId)
    ];

  //  cmercury_log(wp_json_encode($payload));

    $resp = wp_remote_post($endpoint, [
        'headers' => $this->headers(),
        'body'    => wp_json_encode($payload)
    ]);

    if (is_wp_error($resp)) {
        return $resp;
    }

    $code = wp_remote_retrieve_response_code($resp);
    //cmercury_log($code);

    $body = wp_remote_retrieve_body($resp);
    $decoded = json_decode($body, true);

  //  cmercury_log($decoded);

    if ($code !== 200) {
        return new WP_Error('enable_failed', 'Failed enabling attribute');
    }

    return $decoded;
}

	
	//set headers
	private function headers() {
		return ['Content-Type'=>'application/json','X-Token'=>$this->token,'Accept'=>'application/json'];
	}

	
	
}
