<?php
if ( ! defined('ABSPATH') ) exit;

class Cmercury_Mailer {

    public static function init() {
        add_filter('pre_wp_mail', [__CLASS__, 'handle_pre_wp_mail'], 10, 2);
    }

    /**
     * Extract clean email from “Name <email@domain>”
     */
    private static function extract_email($value) {
        if (empty($value)) return '';
        if (preg_match('/<(.+?)>/', $value, $m)) {
            return sanitize_email($m[1]);
        }
        return sanitize_email($value);
    }

    /**
     * Extract name from “Name <email@domain>”
     */
    private static function extract_name($value) {
        if (empty($value)) return '';
        if (preg_match('/^(.*?)</', $value, $m)) {
            return trim(strip_tags($m[1]));
        }
        return '';
    }

    public static function handle_pre_wp_mail($pre_return, $atts) {

        if (get_option('cmercury_enable_mailer', 'no') !== 'yes') {
            return null;
        }

        // allow skipping via header
        if (!empty($atts['headers']) && is_array($atts['headers'])) {
            foreach ($atts['headers'] as $h) {
                if (stripos($h, 'X-CMER-Skip-Mailer') !== false) {
                    return null;
                }
            }
        }

        $to      = isset($atts['to'])      ? $atts['to']      : '';
        $subject = isset($atts['subject']) ? $atts['subject'] : '';
        $message = isset($atts['message']) ? $atts['message'] : '';
        $headers = isset($atts['headers']) ? $atts['headers'] : array();

        $trigger_id = get_option('cmercury_trigger_id','');
        if (empty($trigger_id)) {
            return null;   // no trigger → fallback
        }

        $html = $message;
        $text = wp_strip_all_tags($message);

        $hdr_map = array(
            'FromEmailAddress' => '',
            'ReplyTo'          => '',
            'SenderName'       => '',
        );

        // Parse headers for From / Reply-To
        if (is_array($headers)) {
            foreach ($headers as $h) {

                // FROM
                if (stripos($h, 'From:') === 0) {
                    $raw = trim(preg_replace('/^From:/i', '', $h));
                    $hdr_map['FromEmailAddress'] = self::extract_email($raw);

                    // if display name exists, capture it
                    $name = self::extract_name($raw);
                    if (!empty($name)) {
                        $hdr_map['SenderName'] = $name;
                    }
                }

                // Reply-To
                if (stripos($h, 'Reply-To:') === 0) {
                    $raw = trim(preg_replace('/^Reply-To:/i', '', $h));
                    $hdr_map['ReplyTo'] = self::extract_email($raw);
                }
            }
        }

        // Fallback to stored settings if not found
        if (empty($hdr_map['FromEmailAddress'])) {
            $hdr_map['FromEmailAddress'] = self::extract_email(get_option('cmercury_from_email',''));
        }

        if (empty($hdr_map['ReplyTo'])) {
            $hdr_map['ReplyTo'] = self::extract_email(get_option('cmercury_reply_to',''));
        }

        if (empty($hdr_map['SenderName'])) {
            $hdr_map['SenderName'] = get_option('cmercury_sender_name','');
        }

        // Fire API
        $api    = new Cmercury_Api();
        $result = $api->fire_trigger($to, $trigger_id, $subject, $html, $text, $hdr_map);

        $success = true;

        if (is_wp_error($result)) {
            $success = false;
        } elseif (is_array($result)) {
            foreach ($result as $r) {
                if (is_wp_error($r)) {
                    $success = false;
                    break;
                }
            }
        }

        // If API fails → fallback to native wp_mail()
        if (!$success && get_option('cmercury_mailer_fallback','yes') === 'yes') {
            return null; // let WP handle delivery
        }

        return $success;
    }
}
