<?php
if ( ! defined('ABSPATH') ) exit;

/**
 * class-cmercury-woocommerce.php
 * Class: Cmercury_WooCommerce
 *
 * Option B: Auto-create list + start syncing after "Save WooCommerce Settings"
 *
 * Requirements: Cmercury_Api class for API calls
 */

class Cmercury_WooCommerce {

    const OPTION_LIST_ID        = 'cmercury_wc_list_id';
    const OPTION_AUTO_CREATE    = 'cmercury_wc_auto_create_list';
    const OPTION_AUTO_SYNC      = 'cmercury_wc_auto_sync';
    const OPTION_ATTR_MAP       = 'cmercury_wc_attribute_map';
    const OPTION_LAST_SYNC      = 'cmercury_wc_last_sync';
    const OPTION_FROM_NAME      = 'cmercury_wc_from_name';
    const OPTION_FROM_EMAIL     = 'cmercury_wc_from_email';
    const OPT_SYNC_TOTAL        = 'cmercury_wc_sync_total';
    const OPT_SYNC_DONE         = 'cmercury_wc_sync_done';
    const OPT_SYNC_RUNNING      = 'cmercury_wc_sync_running';
    const OPT_LAST_USER_ID      = 'cmercury_wc_last_user_id';
    const OPT_LAST_ORDER_ID     = 'cmercury_wc_last_order_id';
    const CRON_HOOK             = 'cmercury_wc_cron_sync';
    const POST_SAVE_HOOK        = 'cmercury_wc_post_save_ensure';

    public static function init() {
        add_action('admin_init', [__CLASS__, 'register_settings']);
        add_action('cmercury_render_wc_tab', [__CLASS__, 'render_settings_tab']);

        // AJAX endpoints
        add_action('wp_ajax_cmercury_wc_sync_now', [__CLASS__, 'ajax_sync_now']);
        add_action('wp_ajax_cmercury_wc_sync_status', [__CLASS__, 'ajax_sync_status']);

        // CSV export handler (also hooked by main plugin file - safe to re-hook)
        add_action('admin_post_cmercury_wc_export_csv', [__CLASS__, 'handle_csv_export']);

        // Cron hook
        add_action(self::CRON_HOOK, [__CLASS__, 'cron_sync']);

        // Event handlers for immediate single-customer sync (honor auto_sync)
        add_action('user_register', [__CLASS__, 'on_user_register'], 20, 1);
        if ( class_exists('WooCommerce') ) {
            add_action('woocommerce_thankyou', [__CLASS__, 'on_order_complete'], 20, 1);
        }

        // Post-save single event handler (runs create-list + ensure attrs + schedule cron + schedule immediate sync)
        add_action(self::POST_SAVE_HOOK, [__CLASS__, 'post_save_ensure_handler']);

        // Show a success notice on Save (on admin pages)
       // add_action('admin_notices', [__CLASS__, 'admin_saved_notice']);
    }

    /******************************
     * Settings registration
     ******************************/
    public static function register_settings() {
        register_setting('cmercury_wc_group', self::OPTION_LIST_ID, [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => ''
        ]);

        // Keep these options registered so they exist and can be updated by code later.
        register_setting('cmercury_wc_group', self::OPTION_AUTO_CREATE, [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => 'yes'
        ]);
        register_setting('cmercury_wc_group', self::OPTION_AUTO_SYNC, [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => 'yes'
        ]);
        register_setting('cmercury_wc_group', self::OPTION_ATTR_MAP, [
            'sanitize_callback' => [__CLASS__, 'sanitize_attr_map'],
            'default' => []
        ]);
        register_setting('cmercury_wc_group', self::OPTION_LAST_SYNC, [
            'sanitize_callback' => 'sanitize_text_field',
            'default' => ''
        ]);
        register_setting('cmercury_wc_group', self::OPTION_FROM_NAME, [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => ''
        ]);
        register_setting('cmercury_wc_group', self::OPTION_FROM_EMAIL, [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_email',
            'default' => ''
        ]);
    }

    public static function sanitize_attr_map($val) {
        if (!is_array($val)) return [];
        foreach ($val as $k => $v) { $val[$k] = intval($v); }
        return $val;
    }

    /******************************
     * Render admin settings tab (only List ID is visible)
     ******************************/
    public static function render_settings_tab() {
        // Load options (do NOT auto-save or update options here)
        $list_id            = get_option(self::OPTION_LIST_ID, '');
        $attr_map           = get_option(self::OPTION_ATTR_MAP, []);
        $last_sync          = get_option(self::OPTION_LAST_SYNC, '');
        $sync_total         = intval( get_option(self::OPT_SYNC_TOTAL, 0) );
        $sync_done          = intval( get_option(self::OPT_SYNC_DONE, 0) );
        $sync_running       = get_option(self::OPT_SYNC_RUNNING, 0);
        $auto_sync_status   = get_option(self::OPTION_AUTO_SYNC, 'yes');
        
        if($last_sync==''){
            $txt_last_sync = 'Not started.';
        }
        else {
        $txt_last_sync = esc_html( date_i18n(get_option('date_format').' '.get_option('time_format'), intval($last_sync)) );
        }
        
        if($sync_running==0) {
            $txt_sync_running = 'Not Running.';
        }
        else {
        $txt_sync_running = esc_html( date_i18n(get_option('date_format').' '.get_option('time_format'), intval($sync_running)) );
        }
        
        if (empty(wp_next_scheduled(self::CRON_HOOK))){
        $txt_next_cron =    'Not Scheduled';
        } else {
              $txt_next_cron =  esc_html( date_i18n(get_option('date_format').' '.get_option('time_format'), intval(wp_next_scheduled(self::CRON_HOOK))) );
        }

        // Detect that settings were just saved (WP adds settings-updated=1 on redirect after options.php)
        $settings_saved = isset($_GET['settings-updated']) && ($_GET['settings-updated'] === 'true' || $_GET['settings-updated'] === '1');

        ?>
        <h2>🛒 WooCommerce Customer Sync Settings</h2>

<p class="description">
    Enable automatic synchronization of your WooCommerce customers and their attributes to your cmercury audience list. This data is essential for advanced segmentation and personalized marketing campaigns.
</p>

        <form method="post" action="options.php">
            <?php settings_fields('cmercury_wc_group'); ?>

            <table class="form-table">
                <tr>
                    <th>cmercury Destination List ID</th>
                    <td>
                        <input type="text"
                            name="<?php echo esc_attr(self::OPTION_LIST_ID); ?>"
                            value="<?php echo esc_attr($list_id); ?>"
                            class="regular-text" />
                        <p class="description">
                            Enter your cmercury List Id to sync customers to that list. Leave this field blank to let the plugin create a new list named <strong>Store_Customers</strong> automatically.
                        </p>
                    </td>
                </tr>

            </table>

            <?php submit_button('Save WooCommerce Settings'); ?>
        </form>
        
        <?php if ($list_id != '' ) { 
         echo '<hr><div style="padding:10px 15px; background:#ffffff; border-left:4px solid #51c238; margin-top: 5px; font-weight: normal;">';
         echo '<h3>Current Settings</h3>';
         echo '<h4>List Id: '.$list_id.'</h4>';
         echo '<h4>Auto Sync Enabled: '.ucfirst($auto_sync_status).'</h4>';
         echo '<h4>Approximate Customer Count: '.$sync_total.'</h4>';
        // echo '<h4>Total Synced Count: '.$sync_done.'</h4>';
         echo '<h4>Last Sync at: '.$txt_last_sync.'</h4>';
         echo '<h4>Sync Running from: '.$txt_sync_running.'</h4>';
        // echo '<h4>Current Time: '.esc_html( date_i18n(get_option('date_format').' '.get_option('time_format'), intval(time())) ).'</h4>';
       //  echo '<h4>Next Schedule status: '.$txt_next_cron.'</h4>';
         
         } 
         
        // Check if the attribute map is not empty before rendering the section
        if (!empty($attr_map)) { 
            // 1. Extract the attribute names and format them for display
            $attribute_names = array_keys($attr_map);
            // Convert keys like 'First_Name' to 'First Name' and join them with a comma
            $display_list = implode(', ', array_map(function($name) {
                return str_replace('_', ' ', $name);
            }, $attribute_names));
        ?>
                <h4>Fields Added in List: </h4>
                <?php echo "<code>".esc_html($display_list)."</code>"; ?>
                <p><strong>Note</strong>: The data fields listed above have been added to your list for segmentation and personalization in cmercury.</p>
        <?php }  echo '</div>'; ?>
        
        <hr>

        <?php
        // If settings were saved now AND list id is blank -> schedule post-save ensure
        if ( $settings_saved ) {
            // get the posted value (the form redirect happens before we render; but WP has saved the option by now)
            $saved_list_id = get_option(self::OPTION_LIST_ID, '');
            if ( empty($saved_list_id) ) {
                // schedule a single event to run the post-save ensure handler shortly (avoid doing heavy work during redirect)
                if ( ! wp_next_scheduled(self::POST_SAVE_HOOK) ) {
                    wp_schedule_single_event( time() + 3, self::POST_SAVE_HOOK );
                }

                echo '<div style="padding:10px; background:#fff8e5; border-left:4px solid #f0b400; margin-bottom:10px;">
                        <strong>Settings saved.</strong><br>
                        Preparing your audience list now. This may take a few seconds. The page will refresh shortly. You may check back later too.
                      </div>';
                // client-side refresh to pick up newly created list id after post-save handler runs
                echo "<script>setTimeout(function(){ location.reload(); }, 30000);</script>";
            } else {
                // If settings saved and list id exists, show a simple saved message (admin_notices covers generic)
                echo '<div style="padding:10px; background:#e6ffed; border-left:4px solid #2ecc71; margin-bottom:10px;">
                        <strong>Settings saved.</strong><br>
                        Using List ID: <code>' . esc_html($saved_list_id) . '</code>
                      </div>';
            }
        }
        ?>

        <h3>Manual Full Sync</h3>

        <p>
            Use <strong>Sync Now</strong> to run a full one-time sync of all customers.
        </p>

        <p>
            <button id="cmercury-wc-sync-now" class="button button-primary"<?php echo $sync_running ? ' disabled' : ''; ?>>Sync Now</button>
            <span id="cmercury-wc-sync-status" style="margin-left:12px;"><?php echo $sync_running ? 'Sync running...' : ''; ?></span>
        </p>

        <div id="cmercury-progress" style="margin-top:12px;">
            <?php
                if ($sync_total > 0) {
                    //$pct = round(($sync_done / max(1, $sync_total)) * 100);
                    //echo "Sync Progress: {$sync_done} / {$sync_total} ({$pct}%)";
                    echo "Sync Progressing";
                } elseif ($sync_running) {
                    echo "Sync in progress...";
                } else {
                    echo "";
                }
            ?>
        </div>

        <hr>

        <h3>Export Customers for Manual cmercury Import (CSV) : Optional</h3>
        <form method="post" action="<?php echo admin_url('admin-post.php'); ?>">
            <?php wp_nonce_field('cmercury_wc_export_csv'); ?>
            <input type="hidden" name="action" value="cmercury_wc_export_csv">
            
            <p>
                Use this option to generate a <b>CSV file containing all customer data</b> (registered users and guests who have placed orders) for a one-time manual bulk upload into your cmercury audience list. <br> Note: This step is optional and only needed, if you wish to quickly sync the customers to cmercury.
            </p>
            <p>
                <input type="submit" class="button button-secondary" value="Export Customers">
            </p>
        </form>

        <!-- JS: Start Sync and Polling -->
        <script>
        (function(){
            const btn = document.getElementById('cmercury-wc-sync-now');
            const status = document.getElementById('cmercury-wc-sync-status');
            const progressBox = document.getElementById('cmercury-progress');

            function poll() {
                fetch(ajaxurl, {
                    method: 'POST',
                    credentials: 'same-origin',
                    headers: {'Content-Type':'application/x-www-form-urlencoded'},
                    body: 'action=cmercury_wc_sync_status'
                })
                .then(r => r.json())
                .then(j => {
                    if (!j.success) return;
                    btn.disabled = !!j.data.running;
                    status.innerText = j.data.status_text || '';
                    if (j.data.total) {
                      //  progressBox.innerText = `Sync Progressing...: ${j.data.done} / ${j.data.total} (${j.data.pct}%)`;
                      progressBox.innerText = ``;
                    } else if (j.data.running) {
                        progressBox.innerText = 'Sync in progress...';
                    } else {
                        progressBox.innerText = '';
                    }
                    if (j.data.running) {
                        setTimeout(poll, 30000);
                    } else if (j.data.finished) {
                        status.innerText = 'Sync completed.';
                    } else {
                        // poll again occasionally
                        setTimeout(poll, 30000);
                    }
                })
                .catch(()=>{ setTimeout(poll, 30000); });
            }

            // initial poll
            poll();

            if (!btn) return;
            btn.addEventListener('click', function(e){
                e.preventDefault();
                if (btn.disabled) return;
                btn.disabled = true;
                status.innerText = 'Starting full sync...';

                fetch(ajaxurl, {
                    method: 'POST',
                    credentials: 'same-origin',
                    headers: {'Content-Type':'application/x-www-form-urlencoded'},
                    body: 'action=cmercury_wc_sync_now&_wpnonce=<?php echo wp_create_nonce('cmercury_wc_sync'); ?>'
                })
                .then(r => r.json())
                .then(j => {
                    if (!j.success) {
                        status.innerText = 'Error: ' + (j.data && j.data.message ? j.data.message : 'Failed to start sync');
                        btn.disabled = false;
                        return;
                    }
                    status.innerText = 'Sync started...';
                    poll();
                })
                .catch(err => {
                    status.innerText = 'Error: ' + err;
                    btn.disabled = false;
                });
            });
        })();
        </script>

        <?php
    }


    /******************************
     * Post-save ensure handler
     * (scheduled single event after settings save when list id blank)
     ******************************/
    public static function post_save_ensure_handler() {
        self::log('post_save_ensure_handler started');

        // Ensure from name/email exist (set them now if blank)
        $from_name = get_option(self::OPTION_FROM_NAME, '');
        $from_email = get_option(self::OPTION_FROM_EMAIL, '');

        if ( empty($from_name) ) {
            $from_name = get_bloginfo('name') ?: 'Site';
            update_option(self::OPTION_FROM_NAME, sanitize_text_field($from_name));
            self::log('Set from_name to ' . $from_name);
        }

        if ( empty($from_email) ) {
            $host = wp_parse_url(site_url(), PHP_URL_HOST);
            $host = $host ? preg_replace('/^www\./', '', $host) : 'example.com';
            $guess = 'info@' . $host;
            $guess = sanitize_email($guess);
            update_option(self::OPTION_FROM_EMAIL, $guess);
            self::log('Set from_email to ' . $guess);
        }

        // Ensure auto flags are set to yes (we do this only at post-save time)
        update_option(self::OPTION_AUTO_CREATE, 'yes');
        update_option(self::OPTION_AUTO_SYNC, 'yes');

        // Now attempt to ensure list & attributes, allowing auto-create
        $res = self::ensure_list_and_attributes(true);

        if ( is_wp_error($res) ) {
            self::log('ensure_list_and_attributes failed: ' . $res->get_error_message());
            return;
        }

        // If ensure success and we have a list id, schedule immediate cron run to start syncing
        $list_id = get_option(self::OPTION_LIST_ID, '');
        if ( ! empty($list_id) ) {
            // Prepare sync counters and schedule an immediate cron tick in ~5 seconds
            update_option(self::OPT_SYNC_DONE, 0);
            update_option(self::OPT_SYNC_TOTAL, self::count_total_customers());
            update_option(self::OPT_LAST_USER_ID, 0);
            update_option(self::OPT_LAST_ORDER_ID, 0);
            update_option(self::OPT_SYNC_RUNNING, time());

            // schedule a cron tick shortly (cron frequency will continue afterwards via maybe_schedule_cron())
            if ( ! wp_next_scheduled(self::CRON_HOOK) ) {
                // schedule a one-off immediate call to our cron handler
                wp_schedule_single_event( time() + 5, self::CRON_HOOK );
            } else {
                // If cron already scheduled, still schedule a single immediate run to kickstart sync
                wp_schedule_single_event( time() + 5, self::CRON_HOOK );
            }

            // Also ensure recurring cron is configured if auto_sync enabled
            self::maybe_schedule_cron();

            self::log('post_save_ensure_handler completed — list exists and sync scheduled.');
        }
    }

    /******************************
     * Ensure list & attributes (can auto-create)
     ******************************/
    public static function ensure_list_and_attributes( $allow_auto_create = false ) {
        if (! class_exists('Cmercury_Api') ) {
            return new WP_Error('no_api','Cmercury_Api not available');
        }
        $api = new Cmercury_Api();

        $list_id = get_option(self::OPTION_LIST_ID, '');
        $auto_create = get_option(self::OPTION_AUTO_CREATE, 'yes');

        // Use stored from email/name
        $from_email = get_option(self::OPTION_FROM_EMAIL, '');
        $from_name  = get_option(self::OPTION_FROM_NAME, '');

        // Auto-create decision
        $do_auto = $allow_auto_create || ( $auto_create === 'yes' );

        if ( empty($list_id) && $do_auto ) {
            if ( empty($from_email) ) {
                return new WP_Error('no_from', 'From email is required to create list');
            }

            $res = $api->create_mailing_list($from_email, 'Store_Customers', $from_name ?: 'Site');
            if ( is_wp_error($res) ) return $res;

            if ( is_array($res) && ! empty($res['ListId']) ) {
                $list_id = intval($res['ListId']);
                update_option(self::OPTION_LIST_ID, $list_id);
                self::log("Created list id {$list_id}");
            } else {
                return new WP_Error('create_failed','Failed to create list');
            }
        }

        if ( empty($list_id) ) {
            return new WP_Error('no_list','No list configured');
        }

        // query attributes
        $attr_res = $api->query_attributes($list_id);
        if ( is_wp_error($attr_res) ) return $attr_res;

        // normalize existing
        $existing = [];
        if ( isset($attr_res['Value']) && is_array($attr_res['Value']) ) {
            foreach ($attr_res['Value'] as $a) {
                if (!isset($a['AttributeName'], $a['Id'], $a['Status'])) continue;
                $existing[strtolower($a['AttributeName'])] = [
                    'id' => intval($a['Id']),
                    'status' => strtoupper($a['Status'])
                ];
            }
        }

        $desired = [
            'First_Name'        => 2,
            'Last_Name'         => 3,
            'Country'           => 4,
            'City'              => 9157,
            'State'             => 9158,
            'LastSeenDate'      => 9164,
            'FirstSeenDate'     => 9165,
            'TotalOrders'       => 9166,
            'TotalSpend'        => 9167,
            'AverageOrderValue' => 9168,
            'StoreURL'          => 9173,
            'StoreName'         => 9172,
            'CustomerRoles'     => 9171,
            'StoreUserId'       => 9170
        ];

        $map = get_option(self::OPTION_ATTR_MAP, []);

        foreach ($desired as $raw => $id) {
            $lc = strtolower($raw);
            if (isset($existing[$lc])) {
                $map[$raw] = $existing[$lc]['id'];
                if ($existing[$lc]['status'] === 'N') {
                    $enable = $api->enable_attribute_for_list($list_id, $existing[$lc]['id']);
                    if ( is_wp_error($enable) ) {
                        self::log("Failed enabling attribute {$raw}: " . $enable->get_error_message());
                    }
                }
            } else {
                $map[$raw] = $id;
                $enable = $api->enable_attribute_for_list($list_id, $id);
                if ( is_wp_error($enable) ) {
                    self::log("Failed enabling expected attribute {$raw}: " . $enable->get_error_message());
                }
            }
        }

        update_option(self::OPTION_ATTR_MAP, $map);
        return true;
    }

    /******************************
     * AJAX: Start manual full sync
     ******************************/
    public static function ajax_sync_now() {
        if ( ! check_ajax_referer('cmercury_wc_sync', false, false) ) {
            wp_send_json_error(['message'=>'Invalid nonce'], 400);
        }
        if (! current_user_can('manage_options') ) {
            wp_send_json_error(['message'=>'Unauthorized'], 403);
        }

        // Ensure list & attributes first (allow create)
        $ensure = self::ensure_list_and_attributes(true);
        if ( is_wp_error($ensure) ) {
            wp_send_json_error(['message' => 'Ensure failed: ' . $ensure->get_error_message()]);
        }

        // Reset counters and checkpoints for a fresh run
        update_option(self::OPT_SYNC_DONE, 0);
        update_option(self::OPT_SYNC_TOTAL, self::count_total_customers());
        update_option(self::OPT_LAST_USER_ID, 0);
        update_option(self::OPT_LAST_ORDER_ID, 0);
        update_option(self::OPT_SYNC_RUNNING, time());

        // Run one batch immediately for user feedback
        $res = self::sync_all_customers(1000, 100);

        if ( is_wp_error($res) ) {
            update_option(self::OPT_SYNC_RUNNING, 0);
            self::log('Sync error: ' . $res->get_error_message());
            wp_send_json_error(['message' => 'Sync error: ' . $res->get_error_message()]);
        }

        // ensure recurring cron is scheduled for continuation
        self::maybe_schedule_cron();

        wp_send_json_success(['message' => is_array($res) && isset($res['message']) ? $res['message'] : 'Sync started']);
    }

    /******************************
     * AJAX: Poll sync status
     ******************************/
    public static function ajax_sync_status() {
        if (! current_user_can('manage_options') ) {
            wp_send_json_error(['message'=>'Unauthorized'], 403);
        }
        
        $running = get_option(self::OPT_SYNC_RUNNING, 0);
        $timeout_seconds = 10 * MINUTE_IN_SECONDS;
        
        if ( $running > 0 ) {
            
            // Check if the running timestamp is older than the timeout
            if ( (time() - $running) > $timeout_seconds ) {
                
                // 1. Log the failure (crucial for diagnosis)
                self::log("SYNC WATCHDOG RESET: Detected stalled sync (running since " . date('Y-m-d H:i:s', $running) . "). Resetting state.");
                
                // 2. Perform the hard reset
                update_option(self::OPTION_LAST_SYNC, $running);
                update_option(self::OPT_SYNC_RUNNING, 0); // Clear the running flag
                
                //set cron if not set.
                self::maybe_schedule_cron();
                
                // After reset, we must refetch the 'running' status for the JSON response.
                $running = 0; 
            }
        }
        
        
        $total = intval( get_option(self::OPT_SYNC_TOTAL, 0) );
        $done  = intval( get_option(self::OPT_SYNC_DONE, 0) );

        $is_running = $running ? true : false;
        $pct = $total ? round( ($done / max(1,$total)) * 100 ) : 0;
        $status_text = $is_running ? 'Sync is running...' : ( $done ? 'Last sync complete.' : '' );
        $is_finished = (!$is_running && $done > 0 && $done >= $total);

        wp_send_json_success([
            'running' => $is_running ? 1 : 0,
            'finished' => $is_finished ? 1 : 0,
            'total' => $total,
            'done' => $done,
            'pct' => $pct,
            'status_text' => $status_text
        ]);
    }

    /******************************
     * Main sync worker (batched)
     ******************************/
    public static function sync_all_customers($per_batch = 1000, $pause_ms = 100) {
        global $wpdb;

        if (! class_exists('Cmercury_Api') ) {
            return new WP_Error('no_api','Cmercury_Api not available');
        }

        // refresh running timestamp (heartbeat)
        update_option(self::OPT_SYNC_RUNNING, time());

        $list_id = get_option(self::OPTION_LIST_ID, '');
        if ( empty($list_id) ) {
            update_option(self::OPT_SYNC_RUNNING, 0);
            self::log('Got List ID not configured - From sync_all_customers.');
            return new WP_Error('no_list','List ID not configured');
        }
        
        // ** START FIX FOR ZERO-COUNT SCENARIO **
        $total = intval( get_option(self::OPT_SYNC_TOTAL, 0) );
        if ($total === 0) {
            self::log("Total customer count is zero. Resetting sync state.");
            update_option(self::OPT_LAST_USER_ID, 0);
            update_option(self::OPT_LAST_ORDER_ID, 0);
            update_option(self::OPT_SYNC_DONE, 0);
            update_option(self::OPT_SYNC_RUNNING, 0);
            update_option(self::OPTION_LAST_SYNC, time());
            return ['message' => "Sync complete — no customers to process.", 'processed' => 0];
        }
        // ** END FIX **

        $api = new Cmercury_Api();
        $processed = 0;

        $last_user_id = intval( get_option(self::OPT_LAST_USER_ID, 0) );
        $last_order_id = intval( get_option(self::OPT_LAST_ORDER_ID, 0) );

        // USERS
        $user_ids = $wpdb->get_col( $wpdb->prepare(
            "SELECT ID FROM {$wpdb->users} WHERE ID > %d ORDER BY ID ASC LIMIT %d",
            $last_user_id, $per_batch
        ) );
        
     // If no more users AND no more orders → hard reset
        $order_ids_test = $wpdb->get_col(
            "SELECT ID FROM {$wpdb->posts} WHERE post_type='shop_order' LIMIT 1"
        );
        
        if (empty($user_ids) && empty($order_ids_test)) {
            
            self::log("user id and order id are empty.");
            // fully reset sync
            update_option(self::OPT_LAST_USER_ID, 0);
            update_option(self::OPT_LAST_ORDER_ID, 0);
            update_option(self::OPT_SYNC_RUNNING, 0);
            update_option(self::OPTION_LAST_SYNC, time());
        
            return ['message' => "Nothing to sync — reset completed.", 'processed' => 0];
        }
        
        if (! empty($user_ids)) {
            foreach ($user_ids as $uid) {
                self::log("New Loop begins for {$uid} ({$email})");
                
                $user = get_userdata($uid);
                if (!$user) {
                    update_option(self::OPT_LAST_USER_ID, intval($uid));
                    continue;
                }
                
                $roles = is_array($user->roles) ? $user->roles : [];
                if (! array_intersect($roles, ['customer','subscriber'])) {
                    update_option(self::OPT_LAST_USER_ID, intval($uid));
                    continue;
                }
                
                $email = trim( $user->user_email ?? '' );
                if ( empty($email) || ! is_email($email) ) {
                    update_option(self::OPT_LAST_USER_ID, intval($uid));
                    continue;
                }
                $payload = self::build_customer_payload_from_user($user);
                $r = self::send_customer_to_cmercury($payload);
                if ( is_wp_error($r) ) {
                    self::log("Sync error for user {$uid}: " . $r->get_error_message());
                } else {
                    self::log("Synced user {$uid} ({$email})");
                }

                $processed++;
                
                // Update the heartbeat every 20 records processed
                if ($processed > 0 && $processed % 20 === 0) { 
                    update_option(self::OPT_SYNC_RUNNING, time()); 
                    self::log("Heartbeat update Processed: {$processed}");
                }
                
                
                $done = intval(get_option(self::OPT_SYNC_DONE, 0)) + 1;
                update_option(self::OPT_SYNC_DONE, $done);
                update_option(self::OPT_LAST_USER_ID, intval($uid));

                if ($pause_ms > 0) usleep( (int)$pause_ms * 1000 );
            }

            update_option(self::OPTION_LAST_SYNC, time());
            update_option(self::OPT_SYNC_RUNNING, 0);
            return ['message' => "Processed {$processed} users in this batch", 'processed' => $processed];
        }
        
        $order_ids_test = $wpdb->get_col(
            "SELECT ID FROM {$wpdb->posts} WHERE post_type='shop_order' LIMIT 1"
        );
        
        if (empty($user_ids) && empty($order_ids_test)) {
            
             update_option('user order cross check', 'found empty');
            // fully reset sync
            update_option(self::OPT_LAST_USER_ID, 0);
            update_option(self::OPT_LAST_ORDER_ID, 0);
            update_option(self::OPT_SYNC_RUNNING, 0);
            update_option(self::OPTION_LAST_SYNC, time());
        
            return ['message' => "Nothing to sync — reset completed.", 'processed' => 0];
        }

        // ORDERS (guest)
        $order_post_ids = $wpdb->get_col( $wpdb->prepare(
            "SELECT ID FROM {$wpdb->posts} WHERE post_type = %s AND ID > %d ORDER BY ID ASC LIMIT %d",
            'shop_order', $last_order_id, $per_batch
        ) );

        if (! empty($order_post_ids)) {
            foreach ($order_post_ids as $post_id) {
                $order = wc_get_order($post_id);
                if (! $order) {
                    update_option(self::OPT_LAST_ORDER_ID, intval($post_id));
                    continue;
                }
                $email = trim( $order->get_billing_email() );
                if ( empty($email) || ! is_email($email) ) {
                    update_option(self::OPT_LAST_ORDER_ID, intval($post_id));
                    continue;
                }

                $user = $order->get_user_id() ? get_userdata($order->get_user_id()) : null;
                if ($user) {
                    $payload = self::build_customer_payload_from_user($user, $order);
                } else {
                    $payload = [
                        'email'             => $email,
                        'first_name'        => $order->get_billing_first_name(),
                        'last_name'         => $order->get_billing_last_name(),
                        'FirstSeenDate'     => current_time('mysql'),
                        'TotalOrders'       => 1,
                        'TotalSpend'        => floatval($order->get_total()),
                        'AverageOrderValue' => floatval($order->get_total()),
                        'Country'           => $order->get_billing_country(),
                        'City'              => $order->get_billing_city(),
                        'State'             => $order->get_billing_state(),
                        'LastSeenDate'      => current_time('mysql'),
                        'StoreURL'          => wp_parse_url(site_url(), PHP_URL_HOST) ?? '',
                        'StoreName'         => get_bloginfo('name'),
                        'CustomerRoles'     => 'customer',
                        'StoreUserId'       => 0
                    ];
                }

                $r = self::send_customer_to_cmercury($payload);
                if ( is_wp_error($r) ) {
                    self::log("Sync error for order {$post_id}: " . $r->get_error_message());
                } else {
                    self::log("Synced order {$post_id} ({$email})");
                }

                $processed++;
                
                // Update the heartbeat every 20 records processed
                if ($processed > 0 && $processed % 20 === 0) {
                    update_option(self::OPT_SYNC_RUNNING, time());
                    self::log("Heartbeat update processed: {$processed}");
                }
                
                $done = intval(get_option(self::OPT_SYNC_DONE, 0)) + 1;
                update_option(self::OPT_SYNC_DONE, $done);
                update_option(self::OPT_LAST_ORDER_ID, intval($post_id));

                if ($pause_ms > 0) usleep( (int)$pause_ms * 1000 );
            }

            update_option(self::OPTION_LAST_SYNC, time());
            update_option(self::OPT_SYNC_RUNNING, 0);
            return ['message' => "Processed {$processed} orders in this batch", 'processed' => $processed];
        }

        // Bulk sync is complete. Reset checkpoints and running flag.
        self::log("Bulk Sync Complete. Transitioning to incremental mode.");
        update_option(self::OPT_LAST_USER_ID, 0);
        update_option(self::OPT_LAST_ORDER_ID, 0);
        update_option(self::OPT_SYNC_RUNNING, 0);
        update_option(self::OPTION_LAST_SYNC, time());
        
        return ['message' => "Bulk Sync complete. Transitioning to incremental mode.", 'processed' => $processed];

    }
    
    
    /******************************
     * Incremental sync worker (post-bulk)
     ******************************/
    public static function sync_changed_customers() {
        if (! class_exists('Cmercury_Api') ) {
            return new WP_Error('no_api','Cmercury_Api not available');
        }

        $last_sync_ts = (int) get_option(self::OPTION_LAST_SYNC, 0);
        if ($last_sync_ts === 0) {
            self::log("Incremental Sync skipped: Last sync timestamp not found.");
            return ['message' => "Incremental sync skipped, timestamp missing."];
        }

        // Use a time range starting 1 hour before the last sync time for safety/overlap
        $date_query = date('Y-m-d H:i:s', $last_sync_ts - HOUR_IN_SECONDS);
        self::log("Starting Incremental Sync for changes since: {$date_query}");
        
        $processed = 0;
        $list_id = get_option(self::OPTION_LIST_ID, '');

        // --- A. Sync Changed/New Orders (Crucial for new customers and order data updates) ---
        $orders = wc_get_orders([
            'limit' => -1, // Get all orders in this date range
            'return' => 'ids',
            'date_modified' => '>=' . $date_query, // New or modified orders
            'orderby' => 'date',
            'order' => 'ASC',
        ]);

        foreach ($orders as $order_id) {
            $order = wc_get_order($order_id);
            if (! $order) continue;
            
            // order handler logic 
            self::on_order_complete($order_id); 
            $processed++;
        }

        // --- B. Sync Registered Users (If they haven't placed a recent order, e.g., only registration/profile update) ---
        // Note: WordPress doesn't track 'last profile update' easily, so we rely on the
        // 'user_registered' field and assume order updates cover most activity.
        $user_query = new WP_User_Query([
            'role__in' => ['customer', 'subscriber'],
            'fields' => 'ids',
            'date_query' => [
                'after' => $date_query,
                'column' => 'user_registered',
                'inclusive' => true,
            ],
        ]);
        
        foreach ($user_query->get_results() as $user_id) {
            $user = get_userdata($user_id);
            self::sync_customer_by_user($user);
            $processed++;
        }
        
        // After successful incremental run, update the last sync time to NOW
        update_option(self::OPTION_LAST_SYNC, time());

        return ['message' => "Incremental Sync processed {$processed} changes.", 'processed' => $processed];
    }
    

    /******************************
     * Count total customers (valid emails only)
     ******************************/
    public static function count_total_customers() {
        // Use an associative array to ensure every email is counted only once.
        $unique_emails = [];

        // 1. Count Registered Users (Roles: customer or subscriber)
        $users = new WP_User_Query([
            'role__in' => ['customer','subscriber'],
            'fields' => ['user_email'], // Only retrieve the email for speed
            'number' => -1 // Get all users
        ]);
        
        foreach ($users->get_results() as $user) {
            $email = trim($user->user_email ?? '');
            
            if (! empty($email) && is_email($email)) {
                // Add the email as a key; duplicate keys will be ignored.
                $unique_emails[$email] = 1;
            }
        }

        // 2. Count Guest Customers from Orders
        $orders = wc_get_orders([ 
            'limit' => -1, // Get all orders
            'return' => 'ids' 
        ]);
        
        foreach ($orders as $oid) {
            $o = wc_get_order($oid);
            if (!$o) continue;
            
            // CRITICAL CHECK: Only process orders placed by guests (user_id is 0)
            if (!$o->get_user_id()) {
                $email = trim( $o->get_billing_email() );
                
                if (! empty($email) && is_email($email)) {
                    // Add the guest email. If this guest user was already added 
                    // in Step 1 (e.g., they registered later), the key already exists and no new entry is made.
                    $unique_emails[$email] = 1; 
                }
            }
        }

        // The final count is the total number of unique keys (emails) in the array.
        return count($unique_emails);
    }

    /******************************
     * Send single customer to cmercury (AddContact / UpdateContact)
     ******************************/
    public static function send_customer_to_cmercury($payload) {
        $list_id = get_option(self::OPTION_LIST_ID, '');
        if (empty($list_id)) return new WP_Error('no_list','No List ID');

        if (! class_exists('Cmercury_Api') ) return new WP_Error('no_api','Cmercury_Api not available');

        $api = new Cmercury_Api();
        $map = get_option(self::OPTION_ATTR_MAP, []);
        $contactAttrs = [];

        $static = [
            'Email' => isset($map['Email']) ? $map['Email'] : 1,
            'First_Name' => isset($map['First_Name']) ? $map['First_Name'] : 2,
            'Last_Name' => isset($map['Last_Name']) ? $map['Last_Name'] : 3,
            'Country' => isset($map['Country']) ? $map['Country'] : 4,
        ];

        if (!empty($payload['first_name'])) $contactAttrs[] = ['AttributeId' => strval($static['First_Name']), 'Value' => strval($payload['first_name'])];
        if (!empty($payload['last_name']))  $contactAttrs[] = ['AttributeId' => strval($static['Last_Name']),  'Value' => strval($payload['last_name'])];
        if (!empty($payload['Country']))    $contactAttrs[] = ['AttributeId' => strval($static['Country']),    'Value' => strval($payload['Country'])];

        $custom_keys = ['City','State','LastSeenDate','FirstSeenDate','TotalOrders','TotalSpend','AverageOrderValue','StoreURL','StoreName','CustomerRoles','StoreUserId'];
        foreach ($custom_keys as $k) {
            if (!empty($payload[$k]) && !empty($map[$k])) {
                $contactAttrs[] = ['AttributeId' => strval($map[$k]), 'Value' => strval($payload[$k])];
            }
        }

        $meta = [ 'ContactAttributes' => $contactAttrs ];
        $res = $api->add_contact(intval($list_id), $payload['email'], $meta);
        if ( is_wp_error($res) ) {
            self::log("AddContact failed for {$payload['email']}: " . $res->get_error_message());
            return $res;
        }

        $value_text = '';
        if ( isset($res['Value']) ) {
            if (is_string($res['Value'])) $value_text = $res['Value'];
            elseif (is_array($res['Value']) && isset($res['Value']['Message'])) $value_text = $res['Value']['Message'];
        }

        if ( stripos($value_text, 'already') !== false || stripos($value_text, 'contact already') !== false ) {
            self::log("Contact exists, attempting update for {$payload['email']}");
            $upd = $api->update_contact(intval($list_id), $payload['email'], $meta);
            if ( is_wp_error($upd) ) {
                self::log("UpdateContact failed for {$payload['email']}: " . $upd->get_error_message());
                return $upd;
            }
            return $upd;
        }

        self::log("AddContact success for {$payload['email']}");
        return $res;
    }

    /******************************
     * Build payload from WP_User object
     ******************************/
    public static function build_customer_payload_from_user($user, $order = null) {
        $StoreName = get_bloginfo('name');
        $store_host = wp_parse_url(site_url(), PHP_URL_HOST) ?: '';
        $StoreURL = preg_replace('/^www\./','', $store_host);

        $StoreUserId = $user->ID ?? 0;
        $roles = isset($user->roles) ? $user->roles : [];
        $CustomerRoles = !empty($roles) ? implode('|', $roles) : '';

        $email = $user->user_email ?? '';
        $first = get_user_meta($user->ID,'first_name',true) ?: '';
        $last = get_user_meta($user->ID,'last_name',true) ?: '';

        $last_active = get_user_meta($user->ID, 'last_activity', true) ?: get_user_meta($user->ID,'last_login',true);
        $date_registered = $user->user_registered ?? '';

        $total_orders = 0;
        $total_spend = 0.0;
        if ( class_exists('WC_Customer') ) {
            try {
                $wc_customer = new WC_Customer($user->ID);
                $total_orders = intval( $wc_customer->get_order_count() );
                $total_spend  = floatval( $wc_customer->get_total_spent() );
            } catch (Exception $e) {}
        }

        $country = get_user_meta($user->ID, 'billing_country', true) ?: '';
        $city    = get_user_meta($user->ID, 'billing_city', true) ?: '';
        $state   = get_user_meta($user->ID, 'billing_state', true) ?: '';

        return [
            'email' => $email,
            'first_name' => $first,
            'last_name' => $last,
            'LastSeenDate' => $last_active ?: '',
            'FirstSeenDate' => $date_registered ?: '',
            'TotalOrders' => $total_orders,
            'TotalSpend' => $total_spend,
            'AverageOrderValue' => $total_orders ? round($total_spend / max(1,$total_orders),2) : 0,
            'Country' => $country,
            'City' => $city,
            'State' => $state,
            'StoreURL' => $StoreURL,
            'StoreName' => $StoreName,
            'CustomerRoles' => $CustomerRoles,
            'StoreUserId' => $StoreUserId
        ];
    }

    /******************************
     * CSV export (admin-post)
     ******************************/
    public static function handle_csv_export() {
        if (! current_user_can('manage_options') ) wp_die('Unauthorized');
        check_admin_referer('cmercury_wc_export_csv');

        $filename = 'woocommerce_customers_' . date('Ymd_His') . '.csv';
        $StoreName = get_bloginfo('name');
        $StoreURL = wp_parse_url(site_url(), PHP_URL_HOST) ?: '';

        header("Content-Type: text/csv; charset=UTF-8");
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header("Pragma: no-cache");
        header("Expires: 0");

        $out = fopen('php://output', 'w');

        fputcsv($out, [
            'Email','First Name','Last Name','Country','State','City','LastSeenDate','FirstSeenDate','TotalOrders','TotalSpend','AverageOrderValue','StoreURL','StoreName','CustomerRoles','StoreUserId'
        ]);

        $args = [ 'role__in' => ['customer','subscriber'], 'number' => -1, 'orderby'=>'ID','order'=>'ASC' ];
        $users = (new WP_User_Query($args))->get_results();
        foreach ($users as $user) {
            $payload = self::build_customer_payload_from_user($user);
            if ( empty($payload['email']) || !is_email($payload['email']) ) continue;
            $uid = $user->ID ?? 0;
            $roles = isset($user->roles) ? $user->roles : [];
            $role_out = !empty($roles) ? implode('|', $roles) : '';
            fputcsv($out, [
                $payload['email'],$payload['first_name'],$payload['last_name'],$payload['Country'],$payload['State'],$payload['City'],$payload['LastSeenDate'],$payload['FirstSeenDate'],$payload['TotalOrders'],$payload['TotalSpend'],$payload['AverageOrderValue'],$StoreURL,$StoreName,$role_out,$uid
            ]);
        }

        $orders = wc_get_orders(['limit'=>-1,'orderby'=>'date','order'=>'ASC']);
        foreach ($orders as $order) {
            $email = $order->get_billing_email();
            if ( empty($email) || !is_email($email) ) continue;
            $user = $order->get_user_id() ? get_userdata($order->get_user_id()) : null;
            if ($user) {
                $payload = self::build_customer_payload_from_user($user,$order);
            } else {
                $payload = [
                    'email' => $email,
                    'first_name' => $order->get_billing_first_name(),
                    'last_name' => $order->get_billing_last_name(),
                    'LastSeenDate' => $order->get_date_modified()->date('Y-m-d H:i:s'),
                    'FirstSeenDate' => $order->get_date_created()->date('Y-m-d H:i:s'),
                    'TotalOrders' => 1,
                    'TotalSpend' => floatval($order->get_total()),
                    'AverageOrderValue' => floatval($order->get_total()),
                    'Country' => $order->get_billing_country(),
                    'City' => $order->get_billing_city(),
                    'State' => $order->get_billing_state()
                ];
            }
            fputcsv($out, [
                $payload['email'],$payload['first_name'],$payload['last_name'],$payload['Country'],$payload['State'],$payload['City'],$payload['LastSeenDate'],$payload['FirstSeenDate'],$payload['TotalOrders'],$payload['TotalSpend'],$payload['AverageOrderValue'],$StoreURL,$StoreName,"customer",0
            ]);
        }

        fclose($out);
        exit;
    }

    /******************************
     * Cron scheduling: only schedule if auto-sync enabled and list exists
     ******************************/
    public static function maybe_schedule_cron() {
        
        $auto_sync = get_option(self::OPTION_AUTO_SYNC, 'yes');
        $list_id = get_option(self::OPTION_LIST_ID, '');
        
        // clear existing if auto_sync off
        if ( $auto_sync !== 'yes' ) {
            $ts = wp_next_scheduled(self::CRON_HOOK);
            if ($ts) wp_unschedule_event($ts, self::CRON_HOOK);
            self::log("Cron has been unscheduled.");
            return;
        }
        
        if ( empty($list_id) ) return;
        
        if (! wp_next_scheduled(self::CRON_HOOK) ) {
            $freq = get_option('cmercury_wc_sync_freq', 'cmercury_every_30_minutes') ?: 'cmercury_every_30_minutes';
            wp_schedule_event(time()+60, $freq, self::CRON_HOOK);
            
            $set_next_cron =  esc_html( date_i18n(get_option('date_format').' '.get_option('time_format'), intval(wp_next_scheduled(self::CRON_HOOK))) );
            self::log("Cron has been scheduled for ".$set_next_cron);
        }
    }

    public static function cron_sync() {
        $total = (int) get_option(self::OPT_SYNC_TOTAL, 0);
        $done  = (int) get_option(self::OPT_SYNC_DONE, 0);
        $is_running = get_option(self::OPT_SYNC_RUNNING, 0) > 0;

        // Skip if a manual sync is running (watchdog already handled stuck process)
        if ($is_running) {
            self::log("Cron sync skipped: Sync already running.");
            return;
        }

        // Phase 1: Bulk Sync (Resume/Start)
        if ($done < $total) {
            self::log("Resuming Bulk Sync. Done: {$done} / Total: {$total}");
            $res = self::sync_all_customers(1000, 100);
            $type = 'bulk';

        } 
        // Phase 2: Incremental Sync (Bulk is complete)
        elseif ($total > 0 && $done >= $total) {
            self::log("Bulk Sync complete. Switching to Incremental Sync mode.");
            $res = self::sync_changed_customers();
            $type = 'incremental';
        }
        // Phase 0: No customers (should be rare if count_total_customers is working)
        else {
             self::log("Cron sync skipped: Total customers is 0.");
             return;
        }
        
        if ( is_wp_error($res) ) {
            self::log("Cron {$type} sync error: " . $res->get_error_message());
        } else {
            self::log("Cron {$type} sync: " . (is_array($res) && isset($res['message']) ? $res['message'] : 'ok'));
        }
    }
    
    public static function reschedule_cron( $new_freq ) {
        $timestamp = wp_next_scheduled(self::CRON_HOOK);
        if ($timestamp) {
            wp_unschedule_event($timestamp, self::CRON_HOOK);
        }

        $auto_sync = get_option(self::OPTION_AUTO_SYNC, 'yes');
        $list_id   = get_option(self::OPTION_LIST_ID, '');

        if ($auto_sync !== 'yes' || empty($list_id)) {
            self::log('reschedule_cron: auto_sync disabled or list missing — not scheduling.');
            return;
        }

        wp_schedule_event(time() + 60, $new_freq, self::CRON_HOOK);
        self::log("reschedule_cron: rescheduled with frequency {$new_freq}");
    }

    /******************************
     * Event handlers
     ******************************/
    public static function on_user_register($user_id) {
        if ( get_option(self::OPTION_AUTO_SYNC, 'yes') !== 'yes' ) return;
        $user = get_userdata($user_id);
        if ( ! $user ) return;
        self::sync_customer_by_user($user);
    }

    public static function on_order_complete($order_id) {
        if ( get_option(self::OPTION_AUTO_SYNC, 'yes') !== 'yes' ) return;
        try {
            $order = wc_get_order($order_id);
            if (! $order ) return;
            $user = null;
            if ($order->get_user_id()) $user = get_userdata($order->get_user_id());
            if (! $user) {
                $email = $order->get_billing_email();
                $data = [ 'ID'=>0, 'user_email'=>$email, 'first_name'=>$order->get_billing_first_name(), 'last_name'=>$order->get_billing_last_name() ];
                self::sync_customer_by_array($data, $order);
                return;
            }
            self::sync_customer_by_user($user, $order);
        } catch (Exception $e) {
            self::log('Order handler error: ' . $e->getMessage());
        }
    }

    public static function sync_customer_by_user($user, $order = null) {
        if (! $user ) return;
        $email = $user->user_email ?? '';
        if (! is_email($email)) return;
        $payload = self::build_customer_payload_from_user($user, $order);
        return self::send_customer_to_cmercury($payload);
    }

    public static function sync_customer_by_array($data, $order = null) {
        if ( empty($data['user_email']) || ! is_email($data['user_email']) ) return;
        $payload = [ 'email' => $data['user_email'], 'first_name' => $data['first_name'] ?? '', 'last_name' => $data['last_name'] ?? '' ];
        if ($order) {
            $payload['TotalOrders'] = 1;
            $payload['TotalSpend'] = floatval($order->get_total());
        }
        return self::send_customer_to_cmercury($payload);
    }

    /******************************
     * Simple logger
     ******************************/
    protected static function log($msg) {
        if ( function_exists('cmercury_log') ) {
            cmercury_log($msg);
        } else {
            error_log("cmercury: ".$msg);
        }
    }
}
